Scriptname RF:FuelHandlerQuest Extends Quest

Group Quests
    RF:TravelHandlerQuest Property TravelManager Mandatory Const Auto
    RF:TutorialQuest Property TutorialManager Mandatory Const Auto
    RF:DisasterScript Property DisasterManager Mandatory Const Auto
    SQ_PlayerShipScript Property SQ_PlayerShip Auto
    DialogueShipServicesScript Property DialogueShipServices Auto Const mandatory
EndGroup

Group AVs
    ActorValue Property SpaceshipGravJumpFuel auto const mandatory
    ActorValue Property SpaceshipPartsMass Mandatory Const Auto
    ActorValue Property SpaceshipGravJumpCalculation Mandatory Const Auto
EndGroup

Group System
    Perk Property _RF_Perk Mandatory Const Auto
    { This will hold our SPEL ability to control grav jump range }
    GlobalVariable Property _RF_Sys_ModEnabled Mandatory Const Auto
    { Mod on or off basically }
    GlobalVariable Property _RF_Sys_AllowTravel Mandatory Const Auto
    { Travel System where 1 is enabled }
    GlobalVariable Property _RF_Sys_Verbose Mandatory Const Auto
    { Verbosity where 1 is verbose }
    GlobalVariable Property _RF_EnableCassiopeiaLogic Mandatory Const Auto
    { RFO where 1 gets map data when possible}
    GlobalVariable Property _RF_Sys_TravelLimitHardcore Mandatory Const Auto
    { Travel Limiter where 1 is enabled }
    GlobalVariable Property _RF_Sys_FuelDamageHardcore Mandatory Const Auto
    { Fuel Detonation where 1 is enabled }
    ;DEBUG
    Potion Property DebugPotion1 Mandatory Const Auto
    Potion Property DebugPotion2 Mandatory Const Auto
    Potion Property DebugPotion3 Mandatory Const Auto
    Potion Property DebugPotion4 Mandatory Const Auto
    GlobalVariable Property _RF_DTV Mandatory Const Auto
    { This lets us force certain random events }
    ;States
    GlobalVariable Property _RF_State_CanActivateCargoPanel Mandatory Const Auto
    { This has to be 1 for SmartFuel to fire }
EndGroup

Group Travel

    Location[] Property StarSystemsCore Mandatory Const Auto
    Location[] Property StarSystemsPeriphery Mandatory Const Auto
    Location[] Property StarSystemsOuterRim Mandatory Const Auto
    Keyword Property LocTypeMajorOrbital Mandatory Const Auto
    Keyword Property LocTypeOutpost Mandatory Const Auto
    Keyword Property LocTypeStarSystem Mandatory Const Auto
    Keyword Property LocTypeStarstationExterior Mandatory Const Auto
    Location[] Property UCStarstations Mandatory Const Auto
    Faction Property CrimeFactionCrimsonFleet Mandatory Const Auto
	Faction Property CrimeFactionFreestar Mandatory Const Auto
	Faction Property CrimeFactionParadiso Mandatory Const Auto
	Faction Property CrimeFactionRedMile Mandatory Const Auto
	Faction Property CrimeFactionUC Mandatory Const Auto
    Faction Property SpacerFaction Mandatory Const Auto
    Formlist Property _RF_TwinStarFormlist Mandatory Const Auto
    Keyword Property LocTypeSettledPlanet Mandatory Const Auto
    Keyword Property SpaceshipLinkedInterior Mandatory Const Auto
    Cell[] Property StarStationCells Mandatory Const Auto

EndGroup

Group Ship
    Keyword Property LocTypeSettlement Mandatory Const Auto
    Keyword Property ShipManufacturerDeimos Mandatory Const Auto
    Keyword Property ShipManufacturerHopeTech Mandatory Const Auto
    Keyword Property ShipManufacturerNova Mandatory Const Auto
    Keyword Property ShipManufacturerStroud Mandatory Const Auto
    Keyword Property ShipManufacturerTaiyo Mandatory Const Auto
    Keyword[] Property ShipModuleClasses Mandatory Const Auto
    { 0-3 A-M }
    GlobalVariable[] Property ShipModuleClassMults Mandatory Const Auto
    ConditionForm[] Property AstrodynamicsRank Mandatory Const Auto
    GlobalVariable[] Property AstrodynamicsMults Mandatory Const Auto
    { Match Vanilla skill fuel reductions }
    LocationRefType Property Ship_PilotSeat_RefType Mandatory Const Auto
    { To check if we sat in pilot's seat }
    ConditionForm Property _RF_COND_PlayerShipInterior Mandatory Const Auto
    { I could use ENV_CND_InSealedShip but I want to be sure it's our ship... for now I guess }
    GlobalVariable Property _RF_Val_SiphoningCrimeGoldValue Mandatory Const Auto
    Potion Property _RF_FuelSiphon Mandatory Const Auto
    { I would like to restrict functionality to this item somehow. Maybe greater efficiency? }
    Potion[] Property FuelTanks Mandatory Const Auto
    { In ascending order of quality }
    MiscObject property InorgCommonHelium3 auto const mandatory
    { Only for emergencies! }
    Quest Property MQ101 Mandatory Const Auto
    GlobalVariable Property _RF_State_IsRefuelingOK Mandatory Const Auto
    { This is only set to 1 while using smartfuel now }
    GlobalVariable Property _RF_Drain Mandatory Const Auto
    { should be around 12 with current calcs }
    Keyword Property LocTypeSettledSystem Mandatory Const Auto
EndGroup

Group Messages
    Message Property _RF_Message_FuelRemainingOK Mandatory Const Auto
    { Above 50% at breakpoints }
    Message Property _RF_Message_FuelRemainingLOW Mandatory Const Auto
    { Below 50% above 25% }
    Message Property _RF_Message_FuelRemainingNG Mandatory Const Auto
    { Below 25% }
    Message Property _RF_SiphonFuelWarning Mandatory Const Auto
    Message Property _rf_item_failure Mandatory Const Auto
    Message Property _rf_purchase_complementary Mandatory Const Auto
    Message Property _rf_smartfuel_busy Mandatory Const Auto
    Message Property _rf_smartfuel_failure Mandatory Const Auto
    Message Property _rf_smartfuel_foundhelium Mandatory Const Auto
    Message Property _rf_smartfuel_foundtank Mandatory Const Auto
    Message Property _rf_smartfuel_init Mandatory Const Auto
    Message Property _rf_siphon_begin_generic Mandatory Const Auto
    Message Property _rf_siphon_begin_item Mandatory Const Auto
    Message Property _rf_siphon_begin_oldship Mandatory Const Auto
    Message Property _rf_siphon_fail_cancel Mandatory Const Auto
    Message Property _rf_siphon_fail_generic Mandatory Const Auto
    Message Property _rf_siphon_fail_oldship Mandatory Const Auto
    Message Property _rf_siphon_success Mandatory Const Auto
    Message Property _rf_takeover Mandatory Const Auto
    Message Property _RF_Fueling_Message_starting Mandatory Const Auto
    Message Property _RF_Fueling_Message_OK Mandatory Const Auto
    Message Property _RF_outpostfuel_init Mandatory Const Auto
    Message Property _RF_outpostfuel_finished Mandatory Const Auto
    Message Property _RF_outpostfuel_fail Mandatory Const Auto
    Message Property _RF_StationFuel_confirm Mandatory Const Auto
    Message Property _RF_StationFuel_finished Mandatory Const Auto
    Message Property _RF_StationFuel_fail Mandatory Const Auto
    ;FX
    Spell Property _RF_FX_He3soundeffect Mandatory Const Auto
    WwiseEvent Property SoundWhenTriggeringSmartFuel Mandatory Const Auto
    { Something short and clicky, no big ship noises yet }
    WwiseEvent Property SoundForRawFuelRefill Mandatory Const Auto
    { Should be the helium dispenser sound }
    WwiseEvent Property SoundFuelTankUse Mandatory Const Auto
    { drsc_heliumopen }
    WwiseEvent Property SoundSiphon Mandatory Const Auto
    { Possibly also the fuel tank sound, do I have one on? }
    WwiseEvent Property SoundFAILURE Mandatory Const Auto
    { FailSound() just plays this }
    Message Property _rf_smartfuel_finishedhelium Mandatory Const Auto
    Message Property _rf_smartfuel_finishedtankmultiple Mandatory Const Auto
    WwiseEvent Property SoundStart Mandatory Const Auto
    WwiseEvent Property SoundSuccess Mandatory Const Auto
    Message Property _RF_Fueling_NG_WouldBeWaste Mandatory Const Auto
EndGroup

Actor PlayerRef
SpaceshipReference MyShip

SpaceshipReference SiphonedShip
float SiphonableFuel = 1.0 ; This is updated OnDocking

int StartupTimer = 1

Location[] TwinStars ; Filled on startup from a formlist

; Not using these yet, but we can use a formlist later for these as we refine logic
Location[] SysCore
Location[] SysPeriphery
Location[] SysOuterRim 

float ShipFuelAvailable = 1.0 ; Actual AV amount. Updated in GetFuelLevel     
float ShipMaximumRange = 1.0 ; This is in light years. I probably don't need it
int LastPercent = 100 ; Updated in Restrictions()

bool JumpWasIntrasystem = false ; Updated in GetTravelDistance. True = No Grav Stuff
bool SpoolingUp = false ; Set in GravJumpHandler() in state 0
bool TwinJump = False ; Drain will be random under 20 if this is true
bool MyOldShipSteal = False ; In exactly one scenario a different message will show
bool FirstStart = false ; This will trigger the mod to start OnSit
bool StationDocked = false ; set OnDocking to allow fuel purchasing

int Zone = 0 ; Set in Distancecompare
int CurrentSystemType = 0

bool NewSystemSettled = False ; set in CheckLocationDistance

float MapRealFuel = 0.0 ; This is provided by Cassiopeia if REALFUELON = TRUE 


bool AutoRefuelRunning = false ; state guard for smartfuel

;Counts for breakpoints
int BreakHigh = 70
int BreakNorm = 40
int BreakMerg = 25
int BreakCrit = 10

; System
bool RF = false ; Is the mod running
bool REALFUELON = false ; Is the mod using Cassiopeia as its fuel sender

; DEBUG - DO NOT TOUCH
bool DEBUGON = true ; Logging or QTVB only
bool DEBUGNOTIFY = false ; Prints logs to toast notifs
bool DEBUGSTART = false ; Quickstarts and adds some fuel tanks on init
bool DEBUGCASS = false


;----------------------------------------------------
;----------------- DEBUG AND SYSTEM ----------------- 
;----------------------------------------------------


;Generic debug function that can be disabled for prod
Function QuickTrace(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Trace("RF FuelHandler: " + asTextToPrint)
        IF DEBUGNOTIFY ; Can log in realtime to avoid alt-tabbing constantly
            Debug.Notification("RFUEL: " + asTextToPrint)
        EndIF
    EndIF
EndFunction

Function QuickTraceVB(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Notification("RFUEL: " + asTextToPrint)
    EndIF
EndFunction

;true indicates verbose
bool Function VB()
    bool verbose = false
    if _RF_Sys_Verbose.GetValue() == 1
        verbose = true
    endif
    return verbose
EndFunction

; Greater weight value increases chance of failing flip
bool Function CoinFlip(int aiWeight = 50)
    bool Heads = true
    int Coin = Utility.RandomInt(0, 100)
    If Coin <= aiWeight
        Heads = false
    EndIf
    Return Heads
EndFunction

; Gives us a 0.X > 1.X mult to use for randomization
float Function GetVariance(float afMax = 0.25)
    bool Random = Coinflip()
    float Divergence = 1.0
    float RandomAmount = Utility.RandomFloat(0.01, afMax)
    If Random
        Divergence = 1.0 - RandomAmount
    Else
        Divergence = 1.0 + RandomAmount
    EndIF
    return Divergence
EndFunction

Function HandleSettings()
    int Mode = _RF_Sys_ModEnabled.GetValue() as int
    int Cass = _RF_EnableCassiopeiaLogic.GetValue() as int
    If Mode != 1 ; Mod is disabled in settings
        If RF
            MyShip.RestoreValue(SpaceshipGravJumpFuel, 9999)
            Utility.Wait(4)
            Debug.Notification("Ship Fuel restored to " + GF(MyShip) as int)
            QuickTrace("Mod shutting down")
        EndIf
        RF = False
        FirstStart = True
        PerkRefresh(false)
    Else
        If !RF
            QuickTrace("Mod restarting")
            Dashboard()
        EndIf
        RF = True
        REALFUELON = Cass as bool
        iF REALFUELON
            RegisterCass()
        eNDif
        PerkRefresh(true)
    EndIF
    QuickTrace("HandleSettings ran with Mode: " + Mode + " Cass: " + Cass + " for RF: " + RF + " and RFON: " + REALFUELON)
EndFunction

;True if we are... False otherwise
bool Function ModRunning()
    Return RF
EndFunction

;This returns true if we have at least one
bool Function CheckHold(Form asFormToCheck)
    bool HasItem = false
    If MyShip && asFormToCheck
        int AmountInHold = MyShip.GetItemCount(asFormToCheck)
        If  AmountInHold > 0
            HasItem = True
        EndIf
        QuickTrace("CheckHold ran for " + asFormToCheck + " returning " + AmountInHold )
    EndIf
    Return HasItem
EndFunction

; Returns an item to our inventory or hold. if mod is disabled it prompts an idiot alert
Function HandleReturn(Form asFormToReturn, bool abGoingToHold = false, bool abSilent = false)
    If !abSilent
        FailSound()
    EndIf
    If !abGoingToHold
        PlayerRef.AddItem(asFormToReturn, 1, true)
    Else
        MyShip.AddItem(asFormToReturn, 1, true)
    EndIF
    If !RF
        ;Debug.Notification("This cannot be used right now.")
        _rf_item_failure.show()
    EndIF
EndFunction

; Does exactly what you think it does
Bool Function InMyShip()
    Bool ShipOK = false
    ShipOK = _RF_COND_PlayerShipInterior.IsTrue(PlayerRef) 
    QuickTrace("InMyShip returned " + ShipOK + " with location " + PlayerRef.GetCurrentLocation() ) 
    Return ShipOK
EndFunction

;returns current fuel value of a ship sans strings
float Function GF(SpaceshipReference ShipToCheck)
    Return ShipToCheck.GetValue(SpaceshipGravJumpFuel)
EndFunction

;returns base fuel value of a ship sans strings
float Function GBF(SpaceshipReference ShipToCheck)
    Return ShipToCheck.GetBaseValue(SpaceshipGravJumpFuel)
EndFunction

;Quick refresh - probably not needed but whatever
Function PerkRefresh(bool abEnable)
    IF abEnable
        If !PlayerRef.HasPerk(_RF_Perk)
            PlayerRef.AddPerk(_RF_Perk)
        EndIf
    Else
        If PlayerRef.HasPerk(_RF_Perk)
            PlayerRef.RemovePerk(_RF_Perk)
        EndIf
    EndIF
EndFunction

;To check if we need to be able to take cargo panel actions
Function CheckAltActions()
    int activate = 0
    If LastPercent < BreakHigh
        activate = 1   
    ElseIf MyOldShipSteal || SiphonedShip
        activate = 1
    EndIf
    _RF_State_CanActivateCargoPanel.SetValue(activate)
EndFunction

Function FailSound()
    SoundFAILURE.Play(PlayerREf)
EndFunction

float Function GetHeliumCost()
    Return InorgCommonHelium3.GetGoldValue()
EndFunction


;-------------------------------------------------
;----------------- SYSTEM CHECKS -----------------
;-------------------------------------------------



bool Function is_valid_pair_difference(int x, int y)
    bool pair = false
    int diff = Math.Abs(x - y) as int
    int minVal = Math.Min(x, y) as int
    if diff == 1 && (minVal % 2) == 0
        pair = true
    endif
    ;QuickTrace("pairdiff got " + pair + " from x " + X + " and y " + y)
    return pair
EndFunction

bool Function CheckForTwinStars(Location aOldSys, Location aNewSys)
    bool Pair = False
    int NewSysLoc = TwinStars.Find(aNewSys)
    int OldSysLoc = TwinStars.Find(aOldSys)
    If NewSysLoc + OldSysLoc >= 0 ; This is only true if *both* are twin stars
        Pair = is_valid_pair_difference(NewSysLoc, OldSysLoc)
    EndIF
    ;QuickTrace("CFTS(TM) got " + Pair + " from new " + aNewSys + ": " + NewSysLoc + " and old " + aOldSys + ":" + OldSysLoc)
    Return Pair
EndFunction

;This returns what kind of system we are in
;0:Core 1:Periphery 2:Outer 3: Wastes
int Function GetSysType(Location akLocation)
    int Type = 3
    If StarSystemsCore.Find(akLocation) >= 0
        Type = 0
    ElseIf StarSystemsPeriphery.Find(akLocation) >= 0
        Type = 1
    ElseIf StarSystemsOuterRim.Find(akLocation) >= 0
        Type = 2
    EndIF
    Return Type
EndFunction

; This will create an additive float to push onto DistanceMultFlat 
; This in turn lets us use more or less fuel according to zone transition
; Ranges from 1.2 to 6.2
float Function CompareTypes(int aiO, int aiN)
    float Distance = 0
    float Divergence = 0
    int RealDist = Math.abs(aiN - aiO) as int
    Zone = RealDist * 2
    Divergence = RealDist * ( 1.05 * GetVariance(0.05) ) ; 0 - 3.1
    aiO += 1 ; whoops!
    aiN += 1
    int OXN = aiO * aiN ; 1 - 9
    Distance = ( OXN / 4 ) + 1 ; 1.2 - 5
    float Output = Distance + Divergence ; 1.2 - ~6.2
    QuickTrace("CompareTypes got " + Output + " from Distance " + Distance + " + Disparity " + Divergence + " for Old: " + aiO + " New: " + aiN)
    return Output
EndFunction

; Checks zones and feeds into CompareTypes
; Returns CompareTypes()'s output
float Function CheckLocationDistance(Location aDepartureLocation, Location aArrivalLocation)
    float LocationDistanceMult = 1
    Location[] NewLocArray = aArrivalLocation.GetParentLocations(LocTypeStarSystem)
    Location[] OldLocArray = aDepartureLocation.GetParentLocations(LocTypeStarSystem)
    Location MyNewLoc = NewLocArray[0]
    Location MyOldLoc = OldLocArray[0]
    If MyNewLoc.HasKeyword(LocTypeSettledSystem)
        NewSystemSettled = True
    EndIF
    If MyNewLoc != MyOldLoc
        TwinJump = CheckForTwinStars(MyOldLoc, MyNewLoc)
        If !TwinJump
            int TypeNew = GetSysType(MyNewLoc)
            int TypeOld = GetSysType(MyOldLoc)
            CurrentSystemType = TypeNew
            LocationDistanceMult = CompareTypes(TypeOld, TypeNew)
        EndIF
    EndIf
    QuickTrace("CheckLocationDistance got " + LocationDistanceMult )
    Return LocationDistanceMult
EndFunction

bool Function GetWasJumpIntrasystem(Location akOldLoc, Location akNewLoc)
    Return akOldLoc.IsSameLocation(akNewLoc, LocTypeStarSystem) ; thanks jduvall. Great function
EndFunction

Function SmartNotify(int aiPercent = 120, bool abForce = false)
    If aiPercent > 100 ; indicates failure or default
        aiPercent = GetFuelLevelInt()
    EndIf
    int DisplayPercent = aiPercent
    If !VB() 
        if DisplayPercent < 5
            DisplayPercent = 0
        ElseIf DisplayPercent < 95
            DisplayPercent += ( 5 - ( DisplayPercent % 5) ) ; round off values
        Else
            DisplayPercent = 100
        EndIF
    EndIF
    If aiPercent >= BreakHigh ; skip by default at this level
        If VB() || abForce
            _RF_Message_FuelRemainingOK.Show(DisplayPercent)
        EndIf
    Else
        If aiPercent >= BreakNorm
            _RF_Message_FuelRemainingOK.Show(DisplayPercent)
        Else
            If TravelManager.CheckTravelStatus() || abForce; skip nags if travel is disabled
                If aiPercent >= BreakMerg
                    _RF_Message_FuelRemainingLOW.Show(DisplayPercent)
                Else
                    _RF_Message_FuelRemainingNG.Show(DisplayPercent)
                EndIf
            EndIf
        EndIF
    EndIf
EndFunction


;--------------------------------------------------
;----------------- FUELING SYSTEM -----------------
;--------------------------------------------------


; This is our scripted internal handler to add fuel. 
; Add all of it or add some, that's it. No notifs
; The return value is the clamped amount we used.
int Function Realfuel(bool abComplete, float afAmount = 1.0, bool abSkipDashboard = false)
    Float FuelMissing = GBF(MyShip) - GF(MyShip)
    float FillAmount = 0
    If abComplete
        FillAmount = FuelMissing
    Else
        FillAmount = Math.Clamp(afAmount, 0, FuelMissing)
    EndIf
    MyShip.RestoreValue(SpaceshipGravJumpFuel, FillAmount)
    QuickTrace("REFUELING: Complete: " + abComplete + " for " + FillAmount + " from supplied " + afAmount)
    If !abSkipDashboard ; Only use this for Siphoning
        Dashboard()
    EndIf
    Return FillAmount as int
EndFunction

; This is the only function we actually need lol
Function HandleTravelFuel(float afDistance, float afMapValue = -1.0)

    ; New math will scale Distance as derived LY from 20 > 120 as we go from 1 to 9
    ; It will then multiply this directly by our ShipClassMults of 2, 3.5, 5, 6.5
    ; That's it, times Astrodynamics

    Float FuelUsed = 1.0
    float ADSkill = CheckAstrodynamics()

    If ( !REALFUELON || DEBUGCASS ) || afMapValue <= 0 ; for safety

        float BaseShipMult = (ShipModuleClassMults[ShipModuleClasses.Find(MyShip.GetReactorClassKeyword())] as GlobalVariable).GetValue()

        If TwinJump ; override

            FuelUsed = BaseShipMult * Utility.RandomFloat(0.65, 4.0) ; I think this is good enough
            QuickTrace("HandleTravelFuel Override: TWIN " + FuelUsed )

        Else

            float DistanceDerived = ( ( afDistance * 11 ) + 8 ) * GetVariance(0.025)
            QuickTrace( "HandleTravelFuel - Distance: " + afDistance + " Derived: " + DistanceDerived + " with Class: " + BaseShipMult + " and Skill: " + ADSkill )
            FuelUsed = DistanceDerived * BaseShipMult

        EndIF

    Else

        FuelUsed = afMapValue
        
    EndIf

    float Cap = GBF(MyShip) * 0.8 * GetVariance(0.05) ; Purely for gamefeel
    float FuelDrawn = Math.Clamp(FuelUsed, 1, Cap) * ADSkill
    QuickTrace( "HandleTravelFuel is drawing " + FuelDrawn + " after clamping for total capacity.")
    
    MyShip.DamageValue(SpaceshipGravJumpFuel, ( FuelDrawn - 0.1 ) )

    float Delta = Math.abs( FuelDrawn - MapRealFuel) ; This is just for notifs
    float Dpct = ( Delta / FuelDrawn ) * 100 as int
    QuickTrace("DrawingFuel DELTA = " + Dpct + "% for " + Delta + " from Drawn: " + FuelDrawn + " vs Real: " + MapRealFuel)

    If NewSystemSettled
        float Level = GetFuelLevel()
        If Level < 0.4 && Level > 0.05 ; I don't want this firing if we're dead empty
            float Grace = GBF(MyShip) * 0.1 * GetVariance(0.25)
            Realfuel(false, Grace, true) ; Just so we don't constantly run out of fuel hopping around - mega annoying
        EndIf
    EndIF

    Dashboard()

EndFunction

; Handler function to remove fuel. 
; Does not include Dashboard() as this is typically used for Siphoning or external calls.
; Leaves 1 fuel in tank to avoid divide-by-zero errors
Function ForceDrainFuel(bool abComplete, Float afAmount = 0.0)
    float FuelInTank = CheckFuelAmount()
    float FuelToDrain = 1.0
    If abComplete
        FuelToDrain = FuelInTank
    Else
        if afAmount > 0
            FuelToDrain = Math.Clamp(afAmount, 0, FuelInTank)
        EndIf
    EndIf
    MyShip.DamageValue(SpaceshipGravJumpFuel, ( FuelToDrain - 1 ) )
    QuickTrace("ForceDrained " + FuelToDrain + "from " + FuelInTank) ; No dashboard
EndFunction

bool Function RefuelFromDockedStation()
    bool Fueled = false
    If !StationDocked
        QuickTrace("RFDS cancelling, no station found")
        Return Fueled
    Else
        QuickTrace("RFDS proceeding as we are docked to a station")
        Utility.Wait(1.7)
        float FuelWeNeed = CheckFuelAmount(true)
        float FuelCost = FuelWeNeed * GetHeliumCost()
        int ConfirmFueling = _RF_StationFuel_confirm.Show(FuelCost)
        If ConfirmFueling == 1
            If RealFuel.TryToPayCredits(FuelCost as int) == 0
                Fueled = True
                Utility.Wait(3)
                StartFuelSound()
                Realfuel(true)
                float WaitTime = ( FuelWeNeed * 0.07 ) + 4 ; This would be a little too fast for small amounts so + 4
                Utility.Wait(WaitTime)
                _RF_StationFuel_finished.Show()
                StartFuelSound(false)
            Else
                _RF_StationFuel_fail.Show()
            EndIf
        EndIF
    EndIf
    Return Fueled
EndFunction

bool Function RefuelFromHoldManual()
    Utility.Wait(1.7)
    _rf_smartfuel_init.show()
    float FuelWeNeed = CheckFuelAmount(true)
    int HeliumInHold = MyShip.GetItemCount(InorgCommonHelium3)
    QuickTrace("RFHM thinks we need " + FuelWeNeed + " with available he3 " + HeliumInHold)
    float Efficiency = 0.85 * GetVariance(0.1)
    float HeliumToConsume = Math.Clamp(HeliumInHold, 0, FuelWeNeed)
    float HeliumWeGet = HeliumToConsume * Efficiency
    QuickTrace("RFHM got HeliumWeGet " + HeliumWeGet + " with FuelWeNeed " + FuelWeNeed + " and efficiency" + Efficiency + " using he3 " + HeliumToConsume )
    If HeliumWeGet > 15 ; This is just a nitpick but if it keeps causing problems I will cut it
        Realfuel(false, HeliumWeGet)
        Utility.Wait(4)
        StartFuelSound() ; I will have to do so much testing for these fucking sounds man
        _rf_smartfuel_foundhelium.Show(HeliumToConsume)
        MyShip.RemoveItem(InorgCommonHelium3, HeliumToConsume as int, true)
        ;Game.RequestHUDRolloverRefresh()
        float WaitTime = ( HeliumToConsume * 0.07 ) + 4 ; This would be a little too fast for small amounts so + 4
        Utility.Wait(WaitTime)
        _rf_smartfuel_finishedhelium.show(LastPercent)
        StartFuelSound(false)
        return true
    Else
        QuickTrace("RefuelFromHoldManual failed. Going to RefuelFromHold")
        return false
    EndIF
EndFunction

;Called from our smart activate on the panel
bool Function RefuelFromHold()
    int i = 0
    int Count = 0
    form Tank
    bool Found = False
    int Type = 0
    bool Success = false
        While i < FuelTanks.Length
        IF !Found
            If CheckHold(FuelTanks[i]) ; This will still only trigger our smallest tank.
                Tank = FuelTanks[i] as Form
                Count = MyShip.GetItemCount(Tank)
                Type = i
                Found =  True
            EndIf
        EndIF
        i += 1
    EndWhile
    If Tank
        Utility.Wait(4)
        If GetFuelLevelInt() > BreakNorm ; If it is actually low we check if we can spam. Cool!
            MyShip.RemoveItem(Tank, 1, true)
            PlayerRef.AddItem(Tank, 1, true)
            _rf_smartfuel_foundtank.Show()
            Utility.Wait(3)
            PlayerRef.EquipItem(Tank, false, true) ; ENDCHAIN - no further notifs except in TankHandler.
            Success = True
            QuickTrace("RefuelFromHold successful using " + Tank)
        Else
            ; We want to use as many of these as we need to get to 80%
            float Req =  ( GBF(MyShip) * 0.8 ) - GF(MyShip) ; This should do it
            float Fill = 1.0
            If Type == 0
                Fill = 25
            ElseIF Type == 1
                Fill = 50
            Else
                Fill = 125
            EndIf
            int CanUse = Math.Floor( Req / Fill ) ; Should do the trick
            If Fill / 3 > Req
                _RF_Fueling_NG_WouldBeWaste.Show()
                QuickTrace("RefuelFromHold skipped via overflow logic")
            Else
                int ToUse = Math.Clamp(CanUse, 1, Count) as int ; why the fuck was this 0
                ;SoundFuelTankUse.Play(PlayerRef)
                _rf_smartfuel_foundtank.Show()
                Utility.Wait(2.5)
                StartFuelSound()
                _RF_Fueling_Message_starting.Show(CheckFuelAmount() as int)
                MyShip.RemoveItem(Tank, ToUse) ; Now we just need to add the fuel
                ;Game.RequestHUDRolloverRefresh()
                float WaitTime = ( ToUse * 5 ) + 4 ; This will be 1:1 for 1 and then progressively faster - perfect
                Utility.Wait(WaitTime)
                Realfuel(false, ToUse * Fill, true)
                _rf_smartfuel_finishedtankmultiple.Show(ToUse, CheckFuelAmount() as int) ; ENDCHAIN - no further notifs except in TankHandler.
                StartFuelSound(false)
                Success = True
                QuickTrace("RefuelFromHold successful using " + ToUse + " " + Tank + " for fill " + ToUse * Fill)
            EndIf
        EndIF
    Else
        QuickTrace("RefuelFromHold failed")
    EndIf
    Return Success
EndFunction

Function StartFuelSound(bool abStop = false)
    If abStop
        ;PlayerRef.DispelSpell(_RF_FX_He3soundeffect)
        SoundSuccess.Play(PlayerRef)
    Else
        ;_RF_FX_He3soundeffect.Cast(PlayerREf) 
        SoundFuelTankUse.Play(PlayerRef)
    EndIf
EndFunction

Function SmartFuel(ObjectReference akTargetRef)
    If !AutoRefuelRunning
        QuickTrace("SmartFuel beginning")
        _RF_State_IsRefuelingOK.SetValue(1) ; I don't think this does anything anymore...
        akTargetRef.BlockActivation(true, true)
        SoundWhenTriggeringSmartFuel.Play(PlayerRef)
        AutoRefuelRunning = True
        If GetFuelLevelInt() < BreakHigh || MyOldShipSteal
            bool StationRefueled = RefuelFromDockedStation()
            If !StationRefueled
                If SiphonedShip && !StationDocked
                    SmartFuelSiphon()
                Else
                    bool HadRawFuel = RefuelFromHoldManual()
                    If !HadRawFuel
                        bool HadFuelCanister = RefuelFromHold()
                        If !HadFuelCanister
                            Utility.Wait(3)
                            _rf_smartfuel_failure.show()
                            FailSound()
                        Else
                            Dashboard() ; I don't want to autoclear siphon target
                        EndIf
                    Else
                        Dashboard()
                    EndIf
                EndIf
            EndIf
        EndIf
        Utility.Wait(1)
        AutoRefuelRunning = False
        akTargetRef.BlockActivation(false, false)
        QuickTrace("SmartFuel ending")
    Else
        _rf_smartfuel_busy.Show()
    EndIF
EndFunction


;-------------------------------------------------
;----------------- SHIP CHECKERS -----------------
;-------------------------------------------------


; Quick handler to print our current fuel level
float Function GetFuelLevel()
    float FuelAvailable = 1.0
    float FuelNow = GF(MyShip)
    If FuelNow > 0
        FuelAvailable = FuelNow / GBF(MyShip)
    Else
        MyShip.RestoreValue(SpaceshipGravJumpFuel, Math.abs(FuelNow))
        FuelAvailable = 0
    EndIf
    ShipFuelAvailable = FuelNow
    ;QuickTrace( "PlayerShip Float: " + FuelAvailable + ". Stored: " + ShipFuelAvailable as int)
    return FuelAvailable
EndFunction

int Function GetFuelLevelInt()
    float FuelAvailable = 1.0
    float FuelNow = GF(MyShip)
    float FuelAll = GBF(MyShip)
    If FuelNow > 0
        FuelAvailable = ( FuelNow / ( FuelAll ) * 100 ) as int
    Else
        MyShip.RestoreValue(SpaceshipGravJumpFuel, Math.abs(FuelNow))
        FuelAvailable = 0
    EndIf
    LastPercent = FuelAvailable as int
    QuickTrace( "PlayerShip Percent: " + FuelAvailable as int + "% " + FuelNow as int + "/" + FuelAll as int)
    Return FuelAvailable as int
EndFunction

; Returns float value of the fuel in our tanks.
; If called true, returns float value of empty space in tanks
float Function CheckFuelAmount(bool abAmountMissing = false)
    float Fuel = 1
    If !abAmountMissing
        Fuel =  MyShip.GetValue(SpaceshipGravJumpFuel)
        QuickTrace("Current Fuel: " + Fuel)
    Else
        Fuel = MyShip.GetBaseValue(SpaceshipGravJumpFuel) - MyShip.GetValue(SpaceshipGravJumpFuel)
        QuickTrace("Missing Fuel: " + Fuel)
    EndIF
    Return Fuel
EndFunction

; This should be called before *anything* as there is no real automatic update
SpaceshipReference Function GetShip(String asCallingScript = "Fuel")
    MyShip = SQ_PlayerShip.PlayerShip.GetShipRef()
    ;QuickTrace( "MYSHIP: " + MyShip + " for " + asCallingScript )
    Return MyShip
EndFunction

;This returns a float value to match the vanilla skill % reduction
Float Function CheckAstrodynamics()
    float AmountToLower = 1.0
    int RankFound = 0
    int i = 0
    While i < AstrodynamicsRank.length
        If AstrodynamicsRank[i].IsTrue(PlayerRef)
            AmountToLower = AstrodynamicsMults[i].GetValue()
            RankFound = i
        EndIf
        i += 1
    EndWhile
    ;QuickTrace("CheckAstrodynamics returning " + AmountToLower + " from rank:" + RankFound)
    Return AmountToLower
EndFunction



; AIO handler function to check and clean out all of our systems and values
Function Dashboard()
    GetShip()
    GetFuelLevel()
    GetFuelLevelInt()
    SiphonClear()
    CheckAltActions()
    travelManager.CheckRestrictions()
    TravelManager.SmartLimitHandler(CurrentSystemType)
    DisasterManager.HandleDisasterSystem()
    DialogueShipServices.UpdateFuelGlobals()
    TwinJump = False
    JumpWasIntrasystem = False
    NewSystemSettled = False
    MapRealFuel = 0
    StationDocked = false
    _RF_State_IsRefuelingOK.SetValue(0)
EndFunction


;-------------------------------------------------
;----------------- SYSTEM/EVENTS -----------------
;-------------------------------------------------


Function ImportTwinStars()
    Int listSize = _RF_TwinStarFormlist.GetSize()
    TwinStars = new Location[listSize]
    Int i = 0
    While i < listSize
        TwinStars[i] = _RF_TwinStarFormlist.GetAt(i) as Location
        i += 1
    EndWhile
EndFunction

;This should get our current fuel level and set up our ship
Function HandleModStartup()
    RF = True
    MyShip.AddItem(DebugPotion2, 2, true)
    ImportTwinStars()
    MyShip.RestoreValue(SpaceshipGravJumpFuel, 9999)
    RegisterCass()
    ;This is just Dashboard() in a slightly different order
    GetShip()
    GetFuelLevel()
    GetFuelLevelInt()
    CheckAltActions()
    TravelManager.HandleSystemStartup() ; This is just to kickstart - the only time this needs to change after is from SettingsScript
    travelManager.CheckRestrictions()
    DisasterManager.HandleSystemStartup()
    DialogueShipServices.UpdateFuelGlobals()
    TwinJump = False
    JumpWasIntrasystem = False
    MapRealFuel = 0
    _RF_State_IsRefuelingOK.SetValue(0)
    PerkRefresh(true)
    Debug.Notification("Real Fuel running. " + CheckFuelAmount() as int + " Fuel available.")
EndFunction

Event OnQuestInit()

    StartTimer(7)
    GoToState("Starting")

EndEvent

Event OnTimer(int aiTimerID)

    PlayerRef = Game.GetPlayer()
    RegisterForRemoteEvent(PlayerRef, "OnPlayerLoadGame")
    RegisterForRemoteEvent(PlayerRef, "OnExitShipInterior")
    RegisterForRemoteEvent(PlayerRef, "OnHomeShipSet")
    RegisterForRemoteEvent(PlayerRef, "OnPlayerModifiedShip")
    RegisterForRemoteEvent(PlayerRef, "OnPlayerBuyShip")
    RegisterForRemoteEvent(PlayerREf, "OnSit")
    RegisterForMenuOpenCloseEvent("GalaxyStarMapMenu")
    Debug.Trace("RF started dormant, waiting for OnSit")
    FirstStart = True
    If DEBUGSTART
        ;PlayerRef.AddItem(DebugPotion1, 3, true)
        HandleModStartup()
        MyShip.AddItem(DebugPotion3, 3, true)
        RF = True
        FirstStart = False
        Debug.MessageBox("Real Fuel Debug start triggered!")
    EndIf
    GoToState("")

EndEvent



;-------------------------------------------------
;----------------- TRAVEL CALLS ------------------
;-------------------------------------------------



;This handles just about everything and should be fast and robust (hence arrays only in CheckDistance etc)
Function HandleLocChange(Location akOldLoc, Location akNewLoc, bool WasThereFailure)
    If RF
        QuickTrace("HandleLocChange got Old: " + akOldLoc + ", New: " + akNewLoc)
        If !JumpWasIntrasystem ; This gets set earlier by our ShipTracker.
            If !WasThereFailure
                QuickTrace("HandleLocChange Doing fuel calcs with REALFUELON " + REALFUELON + " and Sender value " + MapRealFuel)
                float TravelDistance = CheckLocationDistance(akOldLoc, akNewLoc)
                HandleTravelFuel(TravelDistance, MapRealFuel)
                Utility.Wait(3)
                SmartNotify()
            Else
                QuickTrace("HandleLocChange: GravDriveFailure! Draining all fuel")
                TravelManager.DamageValueSafe(SpaceshipGravJumpFuel, 0.5)
                Dashboard()
            EndIf
        Else
            Dashboard()
        EndIf
        DisasterManager.HandleDisasterSystem()
    EndIf
EndFunction


Event Actor.OnSit(Actor akSender, ObjectReference akFurniture)
    IF FirstStart
        If akSender == PlayerRef && akFurniture.HasRefType(Ship_PilotSeat_RefType)
            HandleModStartup()
            RF = True
            FirstStart = False
        EndIF
    Else
        If RF
            If akSender == PlayerRef && akFurniture.HasRefType(Ship_PilotSeat_RefType)
                Utility.Wait(3) ; Not sure how long takeover scripts take to run  
                QuickTrace("OnSit detected - comparing ships")
                SpaceshipReference ThisShip = PlayerRef.GetCurrentShipRef()
                If ThisShip != MyShip
                    Takeover(ThisShip, MyShip)
                EndIf
            EndIf
        EndIf
    EndIF
EndEvent


;Mostly just for notifications at this point, I don't need it for logic
Event OnMenuOpenCloseEvent(String asMenuName, Bool abOpening)
    If asMenuName == "GalaxyStarMapMenu" && RF
        If abOpening
            If TravelManager.GetOverheated()
                Debug.Notification("Grav Drive overheated! Repair to continue.")
            Else
                If LastPercent <= BreakCrit
                    TutorialManager.Nag(1)
                Else
                    SmartNotify()
                EndIf
            EndIF
        Elseif !abOpening
            ;Do nothing
        EndIF
    EndIf
EndEvent



;Transfers our old fuel amount to the new ship exactly (overflow is lost)
Event Actor.OnPlayerModifiedShip(Actor akSender, SpaceshipReference akShip)
    QuickTrace("PlayerModifiedShip on firing has current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
    If RF
        float NewFuel = CheckFuelAmount()
        GetShip()
        ForceDrainFuel(true)
        Realfuel(false, ShipFuelAvailable)
        SmartNotify() ; I love this function
        QuickTrace("PlayerModifiedShip after mod has current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
    EndIf
EndEvent



;This is technically different enough from the above that I don't want to merge code
Function HandleShipPurchase(SpaceshipReference akShip)
    If RF
        QuickTrace("HandleShipPurchase fired with current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
        Utility.Wait(4) ; God only knows how long the vanilla jank takes to fire
        float Gratis = ( GF(akShip) * 0.45 * GetVariance() ) + ShipFuelAvailable ; May as well give you some of the old fuel as well
        ForceDrainFuel(true)
        Realfuel(false, Gratis)
        ;Debug.Notification("Purchased ship came with " + FuelInNew + " complementary He-3.")
        _rf_purchase_complementary.show(GF(MyShip) as int)
        QuickTrace("HandleShipPurchase after mod has current fuel " + CheckFuelAmount() + " vs recently stored " + ShipFuelAvailable )
    EndIf
EndFunction



;Fires the above
Event Actor.OnPlayerBuyShip(Actor akSender, SpaceshipReference akShip)
    HandleShipPurchase(akShip)
EndEvent



Event Actor.OnHomeShipSet(Actor akSender, SpaceshipReference akShip, SpaceshipReference akPrevious)
    IF RF
        QuickTrace("Ship changed with previous " + akPrevious + " and current " + akShip)
        Location PlayerIsIn = PlayerRef.GetCurrentLocation()
        If PlayerIsIn.HasKeyword(LocTypeOutpost) || PlayerIsIn.HasKeyword(LocTypeSettlement)
            If akPrevious == akShip
                Utility.Wait(5)
                SmartNotify()
            EndIF
        EndIf
    EndIf
EndEvent



;Sanity checks everything
Event Actor.OnPlayerLoadGame(Actor akSender)
    ModRunning()
    If RF
        QuickTrace("---VM--- Game Load event registered")
        GetShip()
        FirstStart = False ; Safety for updates - may or may not work idk
        RegisterCass()
    Else
        QuickTrace("---VM--- Game Loaded with RF Disabled")
    EndIf
EndEvent



; Event received when a ship initiates and completes docking with a parent
; This will be used for refueling
Function HandleDocking(bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    If RF
        GetShip()
        If myShip == akDocking
            bool IsStarStation = HandleStationDetection(akDocking, akParent)
            IF !IsStarStation
                HandleSiphoningAssignment(akDocking, akParent)
            Else
                ; I can just handle this straight from smartfuel with stationdocked, actually
            EndIF
        EndIF
    EndIf
EndFunction

; This is true if we docked to something registered as a settlement, in theory.
bool Function HandleStationDetection(SpaceshipReference akDocking, SpaceshipReference akParent)
    Cell StationInterior = akParent.GetLinkedCell(SpaceshipLinkedInterior)
    QuickTrace("Checking for StarStation with " + akParent + " and interior " + StationInterior )
    IF StationInterior.HasKeyword(LocTypeSettlement)
        QuickTrace("Interior " + StationInterior + " has LocTypeSettlement!")
        StationDocked = True
    ElseIf StarStationCells.Find(StationInterior) >= 0
        QuickTrace("Found " + StationInterior + " in " + StarStationCells)
        StationDocked = True
    Else
        StationDocked = False
    EndIf
    Return StationDocked
EndFunction

; Sanity check to clear even if no locchange occured
Function HandleUndocking(bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    QuickTrace("Running Siphon info clear on HandleUndocking")
    Dashboard()
    StationDocked = false
    QuickTrace( "SiphonShip should be None:" + SiphonedShip + "with amount 0: " + SiphonableFuel )
EndFunction

; Event received when a ship initiates or completes takeoff
Function HandleShipTakeOff(bool abComplete)
    Dashboard()
EndFunction

; Reassigns our ship to the correct one and sets the old one as a siphon target
Function Takeover(SpaceshipReference TheNewShip, SpaceshipReference MyOldShip)
    float NewShipFuel = Math.Clamp( GF(TheNewShip), 15, GBF(TheNewShip) ) ; Avoid floating errors by adding a negligible amount
    float OldShipFuel = Math.Clamp( GF(MyOldShip), 15, GBF(MyOldShip) )
    QuickTrace("Takeover running on New " + TheNewShip + " having " + NewShipFuel + " vs Old " + MyOldShip + " having" + OldShipFuel)
    Float RandomFillLevel = Utility.RandomFloat(0.35, 0.8)
    float NewShipRoll = Math.Clamp( ( ( NewShipFuel * RandomFillLevel ) * GetVariance(0.15) ), 0, 1000 )
    GetShip()
    If MyShip == TheNewShip
        SiphonedShip = MyOldShip
        SiphonableFuel = OldShipFuel
        MyOldShipSteal = True
        QuickTrace("Takeover successful with " + MyShip + " == " + TheNewShip + ". Draining and setting Siphons as " + SiphonedShip + "with Fuel " + SiphonableFuel)
        ForceDrainFuel(true)
        Realfuel(false, NewShipRoll)
        Utility.Wait(0.75)
        ;Debug.Notification("This ship has " + NewShipRoll as int + " He-3.")
        _rf_takeover.Show(NewShipRoll as int, GBF(MyShip) as int)
    Else
        QuickTrace("TAKEOVER FAIL - EXCEPTION")
    EndIF
EndFunction


;-----------------------------------------------------------
;-------------------- SIPHONING SYSTEM ---------------------
;-----------------------------------------------------------


; Assigned when we dock
Function HandleSiphoningAssignment(SpaceshipReference akDocking, SpaceshipReference akParent)
    SiphonedShip = akParent
    Faction SiphonCrimeFaction = akParent.GetCrimeFaction()
    QuickTrace("Ship crime faction is " + SiphonCrimeFaction)
    float myfuel = GF(akDocking)    
    float theirfuel = GF(akParent)
    float fuel2rand = Math.Clamp( ( ( theirfuel * 0.45 ) * GetVariance(0.4) ), 25, 1000 )
    QuickTrace("Detected dock with akDocking: " + akDocking + " having " + myfuel + "and akParent " + akParent + " having " + theirfuel + " randomized to " + fuel2rand )
    SiphonableFuel = fuel2rand
EndFunction


; Cleared in Dashboard()
Function SiphonClear()
    SiphonableFuel = 0
    SiphonedShip = None
    MyOldShipSteal = False
EndFunction

Function MalwareHandler()
    bool Siphoned = false
    If RF
        If SiphonableFuel > 0 && SiphonedShip
            QuickTrace("Malware running on " + SiphonedShip + " with " + SiphonableFuel + " available.")
            Utility.Wait(0.75)
            int GoAhead = 0
            If !MyOldShipSteal
                GoAhead = _rf_siphon_begin_item.Show(SiphonableFuel)
            Else
                _rf_siphon_begin_oldship.Show()
            EndIF
            If GoAhead == 1
                Siphoned = true
                Siphon(SiphonedShip, SiphonableFuel, true)
                If MyOldShipSteal
                    HandleReturn(_RF_FuelSiphon)
                    _rf_siphon_fail_oldship.Show()
                EndIF
            EndIf
        EndIf
    Else
        HandleReturn(_RF_FuelSiphon) ; This handles notif if mod is disabled
    EndIf
    If !Siphoned ; This fires anytime we did not complete the procedure
        HandleReturn(_RF_FuelSiphon)
        _rf_siphon_fail_generic.Show()
    EndIF
EndFunction

Function SmartFuelSiphon()
    bool Siphoned = false
    If RF
        If SiphonableFuel > 0 && SiphonedShip
            QuickTrace("Siphon running on " + SiphonedShip + " with " + SiphonableFuel + " available.")
            Utility.Wait(0.75)
            int GoAhead = 0
            If !MyOldShipSteal
                _rf_siphon_begin_generic.Show()
                Utility.Wait(3)
                GoAhead = _RF_SiphonFuelWarning.Show()
            Else
                _rf_siphon_begin_oldship.Show()
            EndIF
            If GoAhead == 1
                Siphoned = True
                Siphon(SiphonedShip, SiphonableFuel, false)
            Elseif GoAhead == 0
                _rf_siphon_fail_generic.Show()
            Elseif GoAhead == 2
                Utility.Wait(2)
                Dashboard() ; to clear references - aka uncouple
                _rf_siphon_fail_cancel.Show()
                FailSound()
            EndIf
        EndIf
    EndIf
EndFunction

Function Siphon(SpaceshipReference akShipToSiphon, float afFuelStolen, bool abWithDevice = false)
    QuickTrace("Siphon - beginning with Device: " + abWithDevice)
    StartFuelSound()
    Faction SiphonCrimeFaction = akShipToSiphon.GetCrimeFaction()
    int Withdrawn = Realfuel(false, afFuelStolen, true)
    SiphonedShip.DamageValue( SpaceshipGravJumpFuel, ( Withdrawn - 5 ) )
    float WaitTime = ( Withdrawn * 0.14 ) + 6
    Utility.Wait(WaitTime)
    _rf_siphon_success.show(Withdrawn)
    StartFuelSound(true)
    If !abWithDevice && !MyOldShipSteal
        TriggerBounty(SiphonCrimeFaction)
    EndIf
    Utility.Wait(3)
    Dashboard()
    SmartNotify(LastPercent, true)
EndFunction

; Just pulls either the ship faction or checks location to see if we are plundering a starstation
Function TriggerBounty(Faction akFaction)
    float SiphonPenalty = _RF_Val_SiphoningCrimeGoldValue.GetValue()
    If akFaction && akFaction != SpacerFaction
        akFaction.ModCrimeGold(SiphonPenalty as int, false)
        QuickTrace("Triggered bounty of " + SiphonPenalty + " with " + akFaction)
    Else
        Faction Reporting = CheckShipCrime()
        If Reporting
            Reporting.ModCrimeGold(SiphonPenalty as int, false)
            QuickTrace("Triggered bounty of " + SiphonPenalty + " with " + Reporting)
        Else
            QuickTrace("TriggerBounty did not find faction")
        EndIF
    EndIf
EndFunction

Faction Function CheckShipCrime()
    Faction Controls
    GetShip()
    Location ShipIsIn = MyShip.GetCurrentLocation()
    If UCStarstations.Find(ShipIsIn) > 0
        Controls = CrimeFactionUC
    EndIF
    QuickTrace("Ship is in " + ShipISIn + " for faction check " + Controls)
    Return Controls
EndFunction

;-----------------------------------------------------------
;----------------- CASSIOPEIA FUEL SENDER ------------------
;-----------------------------------------------------------

Function RegisterCass()
    If REALFUELON || DEBUGCASS
        If CassiopeiaPapyrusExtender.GetCassiopeiaPapyrusExtenderVersion() > 0
            bool RegPlot = CassiopeiaPapyrusExtender.RegisterForNativeEvent("RF:FuelHandlerQuest", "OnPlayerPlotRoute")
            Debug.Trace("Cassiopeia Registration ran: " + RegPlot)
        EndIf
    EndIf
EndFunction

Function OnPlayerPlotRoute(ObjectReference akHomeshipRef, Int aeFailedPlotReason, Int aiJumps, Float afShipGravJumpRange, Float afDistance, Float afCargoWeight, Float afCargoCapacity, Float afFuelConsumption, Float afMaxFuel) global
    RF:FuelHandlerQuest RF_FuelHandler = Game.GetFormFromFile(0x00081C, "rbt_RealFuel.esm") as RF:FuelHandlerQuest
    If RF_FuelHandler
        RF_FuelHandler.QuickTrace("PlotRoute: Consumed " + afFuelConsumption + " for " + aiJumps + " jumps across " + afDistance + " LY.")
        RF_FuelHandler.UseSender(afFuelConsumption)
    EndIf
EndFunction

Function UseSender(float afFuelConsumption)
    IF afFuelConsumption
        QuickTrace("CASS: Assigning sender value of " + afFuelConsumption)
        MapRealFuel = afFuelConsumption
    Else
        QuickTrace("CASS EXCEPTION: No fuel sender detected")
    EndIf
EndFunction

;-----------------------------------------------------------
;----------- PERK FRAGMENT CALLS - DO NOT TOUCH ------------
;-----------------------------------------------------------

Function HandleActivateCargoPanel(ObjectReference akTargetRef)
    SmartFuel(akTargetRef)
EndFunction

Function HandleActivateFuelContainer(ObjectReference akTargetRef)
    IF RF
        int He3InContainer = akTargetRef.GetItemCount(InorgCommonHelium3)
        QuickTrace("OUTPOSTFUELING: " + He3InContainer + "He3 available from " + akTargetRef)
        If He3InContainer > (MyShip.GetBaseValue(SpaceshipGravJumpFuel) * 0.10)
            int FuelGotten = Realfuel(false, He3InContainer)
            float WaitTime = ( FuelGotten * 0.07 ) + 4
            akTargetRef.RemoveItem(InorgCommonHelium3, FuelGotten)
            _RF_outpostfuel_init.Show(FuelGotten)
            Game.RequestHUDRolloverRefresh()
            Utility.Wait(WaitTime)
            _RF_outpostfuel_finished.Show(GetFuelLevelInt())
        Else
            Utility.Wait(2)
            _RF_outpostfuel_fail.Show()
        EndIf
    EndIf
EndFunction

; Event received when a ship begins or ends far travel - State { Departure = 0, Arrival = 1 }
; Function HandleFarTravel(Location aDepartureLocation, Location aArrivalLocation, int aState)
;     ; Nothing for now
; EndFunction

; Event that is triggered when fuel has been added to this spaceship
; Check if this triggers with our fuel consumable before hooking up the script. It didn't seem to.
; Function HandleRefuel(int aFuelAdded)
;     QuickTrace("Refuel for " + aFuelAdded + "Detected")
;     DialogueShipServices.UpdateFuelGlobals()
; EndFunction